<?php

namespace App\Http\Controllers;

use App\Http\Requests\QuoteFormRequest;
use App\Http\Resources\QuoteResource;
use App\Jobs\SendQuoteEmail;
use App\Models\ActivityLog;
use App\Repositories\QuoteRepository;
use App\Services\LogService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\ParticipantMail;
use Yoeunes\Toastr\Facades\Toastr;
use Illuminate\Mail\Message;
use App\Models\BusinessCallus; // Model for the new businesscallus table

class QuoteController extends Controller
{
    protected $quoteRepository;
    protected $logService;

    public function __construct(QuoteRepository $quoteRepository, LogService $logService)
    {
        $this->quoteRepository = $quoteRepository;
        $this->logService = $logService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(QuoteFormRequest $request): JsonResponse
    {
        try {
            // Spam kontrolü
            if ($this->quoteRepository->isSpam($request->visit_time)) {
                $this->logService->logWarning(
                    'spam_detected',
                    ['email' => $request->Email],
                    'Quote'
                );

                return response()->json([
                    'message' => 'Spam mesaj algılandı.'
                ], 422);
            }

            // Form verilerini temizle ve kaydet
            $quoteData = [
                'AdSoyad' => strip_tags($request->AdSoyad),
                'SirketAdi' => strip_tags($request->SirketAdi),
                'Telefon' => strip_tags($request->Telefon),
                'Email' => strip_tags($request->Email),
                'Mesaj' => strip_tags($request->Mesaj),
                'visit_time' => $request->visit_time
            ];

            $quote = $this->quoteRepository->createQuote($quoteData);

            // Activity log kaydet
            ActivityLog::create([
                'user_id' => auth()->id(),
                'action' => 'quote_created',
                'model_type' => 'Quote',
                'model_id' => $quote->id,
                'new_data' => $quoteData,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent()
            ]);

            // Başarılı işlem logu
            $this->logService->logInfo(
                'quote_created',
                ['quote_id' => $quote->id],
                'Quote'
            );

            // Email gönderimi için queue kullan
            SendQuoteEmail::dispatch($quoteData);

            return response()->json([
                'message' => 'Mesajınız başarıyla gönderildi.',
                'data' => new QuoteResource($quote)
            ], 201);

        } catch (\Exception $e) {
            // Hata logu
            $this->logService->logError(
                'quote_creation_failed',
                $request->all(),
                'Quote',
                $e
            );
            
            return response()->json([
                'message' => 'Mesajınız gönderilirken bir hata oluştu. Lütfen daha sonra tekrar deneyin.'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
