class CookieConsent {
    constructor() {
        this.cookieName = 'cookie_consent';
        this.consentBanner = null;
        this.init();
    }

    init() {
        if (!this.hasUserConsent()) {
            this.showConsentBanner();
        }
    }

    createConsentBanner() {
        const banner = document.createElement('div');
        banner.className = 'cookie-consent';
        banner.innerHTML = `
            <div class="cookie-consent-message">
                Bu web sitesi, size en iyi deneyimi sunmak için çerezleri kullanmaktadır. 
                Çerezler, web sitemizi nasıl kullandığınızı anlamamıza ve hizmetlerimizi 
                iyileştirmemize yardımcı olur. KVKK ve GDPR kapsamında çerez kullanımı 
                hakkında daha fazla bilgi için 
                <a href="/privacy-policy" style="color: #2196F3;">Gizlilik Politikamızı</a> 
                inceleyebilirsiniz.
            </div>
            <div class="cookie-consent-buttons">
                <button class="cookie-consent-button cookie-consent-settings">Ayarlar</button>
                <button class="cookie-consent-button cookie-consent-decline">Reddet</button>
                <button class="cookie-consent-button cookie-consent-accept">Kabul Et</button>
            </div>
        `;
        return banner;
    }

    showConsentBanner() {
        if (!this.consentBanner) {
            this.consentBanner = this.createConsentBanner();
            document.body.appendChild(this.consentBanner);

            // Event listeners
            this.consentBanner.querySelector('.cookie-consent-accept')
                .addEventListener('click', () => this.acceptCookies());

            this.consentBanner.querySelector('.cookie-consent-decline')
                .addEventListener('click', () => this.declineCookies());

            this.consentBanner.querySelector('.cookie-consent-settings')
                .addEventListener('click', () => this.showSettings());
        }
    }

    acceptCookies() {
        this.setConsent('accepted');
        this.hideConsentBanner();
    }

    declineCookies() {
        this.setConsent('declined');
        this.hideConsentBanner();
        this.disableNonEssentialCookies();
    }

    showSettings() {
        // Çerez ayarları modalını göster
        const settingsHtml = `
            <div class="cookie-settings-modal">
                <h2>Çerez Ayarları</h2>
                <div class="cookie-settings-content">
                    <div class="cookie-setting-item">
                        <label>
                            <input type="checkbox" checked disabled>
                            Gerekli Çerezler
                        </label>
                        <p>Bu çerezler sitenin çalışması için gereklidir ve devre dışı bırakılamaz.</p>
                    </div>
                    <div class="cookie-setting-item">
                        <label>
                            <input type="checkbox" name="analytics" ${this.getConsentForType('analytics') ? 'checked' : ''}>
                            Analitik Çerezler
                        </label>
                        <p>Bu çerezler, ziyaretçilerin siteyi nasıl kullandığını anlamamıza yardımcı olur.</p>
                    </div>
                    <div class="cookie-setting-item">
                        <label>
                            <input type="checkbox" name="marketing" ${this.getConsentForType('marketing') ? 'checked' : ''}>
                            Pazarlama Çerezleri
                        </label>
                        <p>Bu çerezler, size özelleştirilmiş reklamlar göstermek için kullanılır.</p>
                    </div>
                </div>
                <div class="cookie-settings-buttons">
                    <button class="save-settings">Ayarları Kaydet</button>
                </div>
            </div>
        `;
        // Modal gösterme işlemleri
    }

    hideConsentBanner() {
        if (this.consentBanner) {
            this.consentBanner.classList.add('hidden');
            setTimeout(() => {
                this.consentBanner.remove();
                this.consentBanner = null;
            }, 300);
        }
    }

    hasUserConsent() {
        return localStorage.getItem(this.cookieName) !== null;
    }

    setConsent(value) {
        localStorage.setItem(this.cookieName, value);
    }

    getConsentForType(type) {
        const consent = localStorage.getItem(this.cookieName);
        if (consent === 'accepted') {
            return true;
        }
        return false;
    }

    disableNonEssentialCookies() {
        // Analitik ve pazarlama çerezlerini devre dışı bırak
        if (window.ga) {
            window['ga-disable-UA-XXXXXXXX-X'] = true;
        }
        // Diğer üçüncü taraf çerezleri devre dışı bırak
    }
}

// Sayfa yüklendiğinde çerez yöneticisini başlat
document.addEventListener('DOMContentLoaded', () => {
    new CookieConsent();
});
