<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Quote;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Queue;
use App\Jobs\SendQuoteEmail;

class QuoteSubmissionTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Queue::fake();
    }

    /** @test */
    public function it_validates_required_fields()
    {
        $response = $this->postJson('/quote', []);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['AdSoyad', 'SirketAdi', 'Telefon', 'Email', 'Mesaj', 'visit_time']);
    }

    /** @test */
    public function it_validates_email_format()
    {
        $response = $this->postJson('/quote', [
            'AdSoyad' => 'Test User',
            'SirketAdi' => 'Test Company',
            'Telefon' => '5551234567',
            'Email' => 'invalid-email',
            'Mesaj' => 'Test message',
            'visit_time' => time() - 15
        ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['Email']);
    }

    /** @test */
    public function it_creates_quote_and_dispatches_email_job()
    {
        $data = [
            'AdSoyad' => 'Test User',
            'SirketAdi' => 'Test Company',
            'Telefon' => '5551234567',
            'Email' => 'test@example.com',
            'Mesaj' => 'Test message',
            'visit_time' => time() - 15
        ];

        $response = $this->postJson('/quote', $data);

        $response->assertStatus(201)
            ->assertJson([
                'message' => 'Mesajınız başarıyla gönderildi.'
            ]);

        $this->assertDatabaseHas('quotes', [
            'AdSoyad' => $data['AdSoyad'],
            'SirketAdi' => $data['SirketAdi'],
            'Email' => $data['Email']
        ]);

        Queue::assertPushed(SendQuoteEmail::class);
    }

    /** @test */
    public function it_prevents_spam_submissions()
    {
        $data = [
            'AdSoyad' => 'Test User',
            'SirketAdi' => 'Test Company',
            'Telefon' => '5551234567',
            'Email' => 'test@example.com',
            'Mesaj' => 'Test message',
            'visit_time' => time() - 5 // 5 saniye önce
        ];

        $response = $this->postJson('/quote', $data);

        $response->assertStatus(422)
            ->assertJson([
                'message' => 'Spam mesaj algılandı.'
            ]);

        $this->assertDatabaseMissing('quotes', [
            'Email' => $data['Email']
        ]);

        Queue::assertNotPushed(SendQuoteEmail::class);
    }
}
