<?php

namespace Tests\Unit\Controllers;

use Tests\TestCase;
use App\Models\Quote;
use App\Http\Controllers\QuoteController;
use App\Http\Requests\QuoteFormRequest;
use App\Repositories\QuoteRepository;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\JsonResponse;
use Mockery;

class QuoteControllerTest extends TestCase
{
    use RefreshDatabase;

    private $controller;
    private $repository;

    protected function setUp(): void
    {
        parent::setUp();
        $this->repository = Mockery::mock(QuoteRepository::class);
        $this->controller = new QuoteController($this->repository);
    }

    protected function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }

    /** @test */
    public function it_returns_error_for_spam_submissions()
    {
        $request = new QuoteFormRequest();
        $request->merge([
            'visit_time' => time() - 5,
            'AdSoyad' => 'Test User',
            'SirketAdi' => 'Test Company',
            'Telefon' => '5551234567',
            'Email' => 'test@example.com',
            'Mesaj' => 'Test message'
        ]);

        $this->repository->shouldReceive('isSpam')
            ->once()
            ->with($request->visit_time)
            ->andReturn(true);

        $response = $this->controller->store($request);

        $this->assertInstanceOf(JsonResponse::class, $response);
        $this->assertEquals(422, $response->getStatusCode());
        $this->assertEquals('Spam mesaj algılandı.', json_decode($response->getContent())->message);
    }

    /** @test */
    public function it_creates_quote_for_valid_submissions()
    {
        $request = new QuoteFormRequest();
        $request->merge([
            'visit_time' => time() - 15,
            'AdSoyad' => 'Test User',
            'SirketAdi' => 'Test Company',
            'Telefon' => '5551234567',
            'Email' => 'test@example.com',
            'Mesaj' => 'Test message'
        ]);

        $quote = new Quote([
            'AdSoyad' => 'Test User',
            'SirketAdi' => 'Test Company',
            'Telefon' => '5551234567',
            'Email' => 'test@example.com',
            'Mesaj' => 'Test message'
        ]);

        $this->repository->shouldReceive('isSpam')
            ->once()
            ->with($request->visit_time)
            ->andReturn(false);

        $this->repository->shouldReceive('createQuote')
            ->once()
            ->andReturn($quote);

        $response = $this->controller->store($request);

        $this->assertInstanceOf(JsonResponse::class, $response);
        $this->assertEquals(201, $response->getStatusCode());
        $this->assertEquals('Mesajınız başarıyla gönderildi.', json_decode($response->getContent())->message);
    }
}
